  ' -----------------------------------------------------------
  ' PAJ7620U2 Gesture Rec.bas
  ' A program for Micromite to communicate with the CJMCU-7620
  ' Gesture Recognition module (based on the PixArt PAJ7620U2 chip)
  ' for hand gesture sensing and recognition.
  '
  ' Written by Jim Rowe (with help from Tim Blythman) for Silicon Chip.
  ' Last revision 1/9/2021 at 11:20 am

  ' Modified by Kenneth Horton, Warrington England on 4 June 2023.
  ' Following publication in Practical Electronics June 2023
  ' Extensive modifications to original code.
  ' Should work on both Micromite and PicoMite
  ' The module should now be connected to the 5V supply.
  '
  ' Notes:
  ' 1. Communication with the module is via the Micromite's
  ' I2C port, which uses pin 17 for SCL and pin 18 for SDA.
  ' For the Picomite it should be installed on the system bus 
  ' and Option System I2C used to define the pins.

  ' 2. The PAJ7620U2 chip's 7-bit I2C address is specified as 73h.
  ' (MMBasic's I2C function expects a 7-bit I2C address since it adds
  ' the lsb read/write bit automatically; R = 1, W = 0).

  ' 3. When a gesture is recognised by the PAJ7620 it stores the
  ' recognition code for the gesture in Bank0 addresses 43h & 44h.

  ' 4. Address EFh in Bank0 and Bank1 of the PAJ7620 is the
  ' Bank Select address (00 = Bank0, 01h = Bank1)

  ' 5. The PAJ7620 chip's ID information is stored in Bank0 addresses
  ' 00, 01 and 02h, with address 01 holding 76h, address 00 holding 20h
  ' and address 02 holding the version number (usually 01h). These
  ' addresses are read-only, and used by the program to check if
  ' the PAJ7620 chip is present and active.
  ' ----------------------------------------------------------
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  Option Default Integer
  
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  Const MaxString = 17          ' Maximum length of gesture string
  Const True = 1
  Const False = 0
  Const DisplayTime = 1000      ' The time the gesture is shown on screen (mS)
  
  DIM IDbytes(2)                ' array to hold PAJ7620 chip ID bytes 0-2 & Gestures 0-1
  DIM GString$ Length MaxString = space$(MaxString)' blank until gesture is recognised
  DIM AddrByte                  ' 8 bits = register address
  DIM Idata                     ' 8 bits = initialising data
  DIM GData                     ' to store gesture flag register 43h and 44h contents
  Dim Timeout                   ' timeout to blank screen between gestures

  ' enable the Micromite I2C module in master mode
  ' (speed = 100kHz, timeout = 200 milliseconds)
  if left$(MM.DEVICE$,4) <> "Pico" then I2C OPEN 100, 200
  
  ' ---------------------------------------------------------------------
  ' start of main part of program
  PAUSE 1                         ' wait 1ms for PAJ7620 to stabilise after power up
  ShowStUpScrn                    ' now show the startup screen
  if PAJ7620Init() <> False then  ' then initialise the PAJ7620 device
    PRINT "PAJ7620 initialised..."
    PAUSE 2000
    PRINT "Start gesture input:"  ' invite user to input gestures
    ShowOpScrn                    ' and show operating screen
  
    ' main program loop starts here
    DO
      ' first read PAJ7620 register 43h and 44h in Bank0 (Gesture flag registers)
      ReadGReg
      IF GData <> 0 THEN          ' if GData is other than zero
        GoReadGesture             ' go decode GData & find GString$
        TEXT 10, MM.VRes*3/8, GString$, LM, 1,2, White,Black  ' display
        PRINT GString$            ' print as well on PC serial monitor screen
        Timeout = timer
      ELSE                        ' but if GByte = 0
        if (left$(GString$,1) <> " ") and (Timer - Timeout > DisplayTime)  then
          GString$ = space$(MaxString)
          TEXT 10, MM.VRES*3/8, GString$, LM, 1,2, WHITE,BLACK 'blank display
        end if
      ENDIF
    LOOP
  end if
' end of main program, subroutines follow
' **********************************************************************

  ' subroutine to display the startup screen
SUB ShowStUpScrn
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT 10, MM.VRes/8, "SILICON CHIP's", LM, 1,2, Green, Black
  if left$(MM.DEVICE$,4) = "Pico" then 
    TEXT 10, MM.VRes/4, "Picomite Program", LM, 1,2, Green, Black
  else
    TEXT 10, MM.VRes/4, "Micromite Program", LM, 1,2, Green, Black
  end if
  TEXT 10, MM.VRes*3/8, "for trying out the", LM, 1,2, Green, Black
  TEXT 10, MM.VRes/2, "CJMCU Hand Gesture", LM, 1,2, Green, Black
  TEXT 10, MM.VRes*5/8, "recognition module", LM, 1,2, Green, Black  
  TEXT 10, MM.VRes*3/4, "Modified by K Horton", LM, 1,2, Green, Black  
END SUB
  ' -----------------------------------------------------------------
    ' subroutine to initialise the PAJ7620
Function PAJ7620Init()
  I2C WRITE &H73,0,2,&HEF,&H00          ' send 00 to register EFh - Wake up
  I2C WRITE &H73,0,2,&HEF,&H00          ' then send again, to select Bank0

  ' See if PAJ7620 is present & functioning, get its ID
  I2C WRITE &H73,1,1,0                  ' first write first address to read from (00)
  I2C READ &H73,0,3, IDbytes()          ' then get the three ID bytes
  PRINT "Device ID is: ", HEX$(IDbytes(1),2), HEX$(IDbytes(0),2)," V." HEX$(IDbytes(2),2)
  IF (IDbytes(0) <> &H20) or (IDbytes(1) <> &H76) THEN
    PRINT "Error: PAJ7620 not found!"
    PAJ7620Init = False
    exit Function
  end if

  do
    Read AddrByte, IData                 ' Read the initialisation data
    if AddrByte < 0 then exit do         ' All done
    I2C WRITE &H73,0,2, AddrByte, IData  ' then send it to the PAJ7620
  loop
  PAJ7620Init = True
End Function
  ' -----------------------------------------------------------------
  ' subroutine to display the operating screen
SUB ShowOpScrn
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT 10, MM.VRes/8, "Gesture Sensed:", LM, 1,2, RED, Black
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to read Gesture registers (Bank0, 43h and 44h)
SUB ReadGReg
  I2C Write &H73,1,1,&H43   ' write register address so we
  I2C READ &H73,0,2, IDbytes() ' can read data from it, then
  GData = IDbytes(0) or (IDbytes(1) << 8) ' Concatenate the registers
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to display the gesture recognised
SUB GoReadGesture
  SELECT CASE GData
    CASE &H01           ' 01h in Reg 43h = Left Flag
      GString$ = "Left"
    CASE &H02           ' 02h in Reg 43h = Right Flag
      GString$ = "Right"
    CASE &H04           ' 04h in Reg 43h = Down Flag
      GString$ = "Down"
    CASE &H08           ' 08h in Reg 43h = Up Flag  
      GString$ = "Up"
    CASE &H10           ' 10h in Reg 43h = Forward Flag
      GString$ = "Forward"
    CASE &H20           ' 20h in Reg 43h = Backward Flag
      GString$ = "Backward"
    CASE &H40           ' 40h in Reg 43h = Clockwise Flag
      GString$ = "Clockwise"
    CASE &H80           ' 80h in Reg 43h = Counter clockwise Flag
      GString$ = "Counter Clockwise"
    CASE &H100          ' 01h in Reg 44h = Wave Flag
      GString$ = "Wave"
    END SELECT
  if len(GString$) < MaxString then GString$ = GString$ + space$(MaxString - len(GString$))
END SUB

' Initialisation data
Data &HEF, &H00   ' send 00h TO ADDR EFh, to SWITCH TO BANK 0 <<*********
Data &H37, &H07   ' addr 37h = R_CursorClampLeft(4:0)
Data &H38, &H17   ' addr 38h = R_CursorClampRight(4:0)
Data &H39, &H06   ' addr 39h = R_CursorClampUp(4:0)
Data &H42, &H01   ' addr 42h = R_Int_2_En(7:0)
Data &H46, &H2D   ' addr 46h = R_AELedOff_UB(7:0)
Data &H47, &H0F   ' addr 47h = R_AELedOff_LB(7:0)
Data &H48, &H3C   ' addr 48h = R_AE_Exposure_UB(7:0)
Data &H49, &H00   ' addr 49h = R_AE_Exposure_UB(15:8)
Data &H4A, &H1E   ' addr 4Ah = R_AE_Exposure_LB(7:0)
Data &H4C, &H20   ' addr 4Ch = R_AE_Gain_UB(7:0)
Data &H51, &H10   ' addr 51h = R_Manual_GG(0)/R_Manual_Exposure(1) etc...
Data &H5E, &H10   ' addr 5Eh = R_Other_CLK_manual
Data &H60, &H27   ' addr 60h = TS_osc_code(6:0)/OSC_BIST_OK(7)
Data &H80, &H42   ' addr 80h = Im_GPIO3
Data &H81, &H44   ' addr 81h = Tm_GPIO3_OEL/Tm_GPIO3_IEB/R_GPIO3
Data &H82, &H04   ' addr 82h = Im_INT/Tm_INT_OEL/Tm_INT_IEB/Tm_INT
Data &H8B, &H01   ' addr 8Bh = R_Cursor_ObjectSizeTh(7:0)
Data &H90, &H06   ' addr 90h = R_NoMotionCountThd(6:0)
Data &H95, &H0A   ' addr 95h = R_ZDirectionThd(4:0)
Data &H96, &H0C   ' addr 96h = R_ZDirectionXYThd(4:0)
Data &H97, &H05   ' addr 97h = R_ZDirectionAngleThd(3:0)
Data &H9A, &H14   ' addr 9Ah = R_RotateXYThd(4:0)
Data &H9C, &H3F   ' addr 9Ch = R_FilterWeight(1:0)/R_FilterDistThd(4:0)
Data &HA5, &H19   ' addr A5h = R_FilterImage/FilterAverage_Mode/R_UseLightWeight
Data &HCC, &H19   ' addr CCh = R_YtoZSum(5:0)
Data &HCD, &H0B   ' addr CDh = R_YtoZFactor(5:0)
Data &HCE, &H13   ' addr CEh = R_PositionFilterLength/ProcessFilterLength(2:0)
' Data &HCF, &H64   ' addr CFh = R_WaveCountThd(3:0)/WaveAngleThd(7:4)
' This makes the wave gesture a little more responsive!
Data &HCF, &H62   ' addr CFh = R_WaveCountThd(3:0)/WaveAngleThd(7:4)
Data &HD0, &H21   ' addr D0h = R_AbortCountThd(2:0)/AbortXYRatio(4:0)

Data &HEF, &H01   ' send 01h TO ADDR EFh, to SWITCH TO BANK 1 <<*********
Data &H02, &H0F   ' addr 02h = Cmd_HStart (5:0)
Data &H03, &H10   ' addr 03h = Cmd_VStart (5:0)
Data &H04, &H02   ' addr 04h = Cmd_ASkip_V/ASkip_H/DAvg_V/VFlip/HFlip
Data &H25, &H01   ' addr 25h = R_LensShadingComp_EnH
Data &H27, &H39   ' addr 27h = R_OffsetY (6:0)
Data &H28, &H7F   ' addr 28h = R_LSC (6:0)
Data &H29, &H08   ' addr 29h = R_LSFT (3:0)
Data &H3E, &HFF   ' addr 3Eh = Cmd_DebugPettern (7:0)
Data &H5E, &H3D   ' addr 5Eh = T_clamp_drv_ctl (1:0)
Data &H65, &H96   ' addr 65h = R_IDLE_TIME (7:0)
Data &H67, &H97   ' addr 67h = R_IDLE_TIME_SLEEP_1 (7:0)
Data &H69, &HCD   ' addr 69h = R_IDLE_TIME_SLEEP_2 (7:0)
Data &H6A, &H01   ' addr 6Ah = R_IDLE_TIME_SLEEP_2 (15:8)
Data &H6D, &H2C   ' addr 6Dh = R_Obj_TIME_2 (7:0)
Data &H6E, &H01   ' addr 6Eh = R_Obj)TIME_2 (15:8)
Data &H72, &H01   ' addr 72h = R_TG_EnH
Data &H73, &H35   ' addr 73h = R_AUTO_SLEEP_Mode1/Mode2/Wakeup_Mode
Data &H74, &H00   ' addr 74h = R_WakeUpSig_Sel (1:0)
Data &H77, &H01   ' addr 77h = R_SRAM_Read_EnH

Data &HEF, &H00   ' send 00h TO ADDR EFh, to SWITCH TO BANK 0 <<*********
Data -1  , -1     ' End of data


